/**
 * 
 */
package gov.va.med.mhv.usermgmt.util;

import java.util.Collection;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.mhv.core.util.DescriptionBuilder;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.usermgmt.enumeration.AccessDomain;
import gov.va.med.mhv.usermgmt.enumeration.AccessPermission;
import gov.va.med.mhv.usermgmt.transfer.AccessControl;
import gov.va.med.mhv.usermgmt.transfer.AccessFeature;
import gov.va.med.mhv.usermgmt.transfer.AccessRole;
import gov.va.med.mhv.usermgmt.transfer.AccessRoleAccessControl;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileAccessControl;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class AccessControlUtils {
    
    private AccessControlUtils () {
    }
    private static final Log LOG = LogFactory.getLog(AccessControlUtils.
        	class);
    
    public static boolean hasAccess(AccessDomain domain, String featureName, 
        AccessPermission permission, AccessControl accessControl) 
    {
        return (accessControl != null)  
            && (hasFeature(domain, featureName, accessControl))
            && AccessPermissionUtils.hasPermission(permission, 
                    accessControl.getPermissions());
    }

    public static boolean hasAccess(AccessDomain domain, String featureName, 
        AccessPermission permission, Collection<AccessControl> accessControls) 
    {
    	LOG.debug("Entered AccessControlUtils::hasAccess");
        if (accessControls != null) {
        	LOG.debug("accessControls != null");
            for (AccessControl accessControl: accessControls) {
                if (hasFeature(domain, featureName, accessControl)) {
                	LOG.debug("Entered hasAccess::hasFeature");
                    return AccessPermissionUtils.hasPermission(
                        permission, accessControl.getPermissions());
                }
            }
        }
        
        return false;
    }

    public static boolean hasAccess(AccessFeature feature, 
        AccessPermission permission, AccessControl accessControl) 
    {
        return (accessControl != null)  
            && (hasFeature(feature, accessControl))
            && AccessPermissionUtils.hasPermission(permission, 
                    accessControl.getPermissions());
    }

    public static boolean hasAccess(AccessFeature feature, 
        Long permissions,
        AccessControl accessControl) 
    {
        return (accessControl != null)  
            && (hasFeature(feature, accessControl))
            && AccessPermissionUtils.hasPermissions(permissions, 
                    accessControl.getPermissions());
    }

    public static boolean hasAccess(AccessFeature feature, 
        AccessPermission permission,
        Collection<AccessControl> accessControls) 
    {
        if (accessControls != null) {
            for (AccessControl accessControl: accessControls) {
                if (hasFeature(feature, accessControl)) {
                    return AccessPermissionUtils.hasPermission(
                        permission, accessControl.getPermissions());
                }
            }
        }
        
        return false;
    }

    public static boolean hasAccess(AccessFeature feature, 
        Long permissions, Collection<AccessControl> accessControls) 
    {
        if (accessControls != null) {
            for (AccessControl accessControl : accessControls) {
                if (hasFeature(feature, accessControl)) {
                    return AccessPermissionUtils.hasPermissions(
                        accessControl.getPermissions(), permissions);
                }
            }
        }
        return false;
    }

    public static boolean hasAccess(AccessFeature feature, 
        AccessPermission[] permissions, 
        Collection<AccessControl> accessControls) 
    {
        if (accessControls != null) {
            for (AccessControl accessControl : accessControls) {
                if (hasFeature(feature, accessControl)) {
                    return AccessPermissionUtils.hasPermissions(
                        accessControl.getPermissions(), permissions);
                }
            }
        }
        return false;
    }


    public static boolean hasFeature(AccessFeature feature, 
        AccessControl accessControl)
    {
        Precondition.assertNotNull("feature", feature);   
        Precondition.assertNotNull("accessControl", accessControl);   
        Precondition.assertNotNull("accessControl.feature", accessControl.
            getFeature());   
       return hasFeature(feature.getDomain(), feature.getName(), accessControl);
    }

    public static boolean hasFeature(AccessDomain domain, String featureName, 
        AccessControl accessControl)
    {
    	LOG.debug("Entered hasFeature()");
        Precondition.assertNotNull("domain", domain);   
        Precondition.assertNotBlank("featureName", featureName);   
        Precondition.assertNotNull("accessControl", accessControl);   
        Precondition.assertNotNull("accessControl.feature", accessControl.
            getFeature());
        LOG.debug("accessControl.getFeature().getDomain(): " + accessControl.getFeature().getDomain());
        LOG.debug("domain: " + domain);
        LOG.debug("accessControl.getFeature().getName(): " + featureName);
       return 
           accessControl.getFeature().getDomain().equals(domain)
           && accessControl.getFeature().getName().equals(featureName);
    }

    public static boolean isForRole(AccessRole role, 
        AccessControl accessControl)
    {
        Precondition.assertNotNull("role", role);   
        Precondition.assertNotNull("role.id", role.getId());   
        Precondition.assertNotNull("accessControl", accessControl);   
        Precondition.assertNotNull("accessControl.roleId", accessControl.
            getRoleId());   
       return role.getId().equals(accessControl.getRoleId());
    }

    public static boolean isForUserProfile(UserProfile userProfile, 
        AccessControl accessControl)
    {
        Precondition.assertNotNull("userProfile", userProfile);   
        Precondition.assertNotNull("userProfile.id", userProfile.getId());   
        Precondition.assertNotNull("accessControl", accessControl);   
        Precondition.assertNotNull("accessControl.roleId", accessControl.
            getUserProfileId());   
       return userProfile.getId().equals(accessControl.getUserProfileId());
    }

    public static String describe(Collection<AccessControl> accessControls) {
        StringBuilder builder = new StringBuilder();
        builder.append(DescriptionBuilder.Properties.COLLECTION_OPEN);
        if (accessControls != null) {
            boolean first = true;
            for (AccessControl accessControl: accessControls) {
                if (first) {
                    first = false;
                } else {
                    builder.append(DescriptionBuilder.Properties.SEPARATOR);
                }
                builder.append(describe(accessControl));
            }
        }
        builder.append(DescriptionBuilder.Properties.COLLECTION_CLOSE);
        return builder.toString();
    }

    public static String describe(AccessControl accessControl) {
        StringBuilder builder = new StringBuilder();
        if (accessControl != null) {
            builder.append("AccessControl");
            builder.append(Integer.toHexString(accessControl.hashCode()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_OPEN);
            builder.append("feature=");
            builder.append(describe(accessControl.getFeature()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
            builder.append("permissions=");
            builder.append(AccessPermissionUtils.describe(accessControl.
                getPermissions()));
            if (accessControl.getRoleId() != null) {
                builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
                builder.append("roleId=");
                builder.append(accessControl.getRoleId());
            }
            if (accessControl.getUserProfileId() != null) {
                builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
                builder.append("userProfileId=");
                builder.append(accessControl.getUserProfileId());
            }
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_CLOSE);
        } else {
            builder.append("<No AccessControl>");
        }
        return builder.toString();
    }

    public static String describe(AccessRoleAccessControl accessControl) {
        StringBuilder builder = new StringBuilder();
        if (accessControl != null) {
            builder.append("AccessRoleAccessControl");
            builder.append(Integer.toHexString(accessControl.hashCode()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_OPEN);
            builder.append("feature=");
            builder.append(describe(accessControl.getFeature()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
            builder.append("permissions=");
            builder.append(AccessPermissionUtils.describe(accessControl.
                getPermissions()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
            builder.append("role=");
            builder.append(describe(accessControl.getRole()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_CLOSE);
        } else {
            builder.append("<No AccessRoleAccessControl>");
        }
        return builder.toString();
    }

    public static String describe(UserProfileAccessControl accessControl) {
        StringBuilder builder = new StringBuilder();
        if (accessControl != null) {
            builder.append("UserProfileAccessControl");
            builder.append(Integer.toHexString(accessControl.hashCode()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_OPEN);
            builder.append("feature=");
            builder.append(describe(accessControl.getFeature()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
            builder.append("permissions=");
            builder.append(AccessPermissionUtils.describe(accessControl.
                getPermissions()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_SEPARATOR);
            builder.append("userProfile=");
            builder.append(describe(accessControl.getUserProfile()));
            builder.append(DescriptionBuilder.Properties.PROPERTY_LIST_CLOSE);
        } else {
            builder.append("<No UserProfileAccessControl>");
        }
        return builder.toString();
    }

    public static String describe(AccessFeature feature) {
        return (feature != null)
            ? feature.getDomain()  + "::" + feature.getName()
            : "<No feature>";
    }

    public static String describe(AccessRole role) {
        return (role != null) 
            ? role.getName() + " (isDefaultPatientRole=" 
                + BooleanUtils.isTrue(role.getIsDefaultPatientRole()) + ")" 
            : "<No role>";
    }

    private static String describe(UserProfile userProfile) {
        return (userProfile != null) ? userProfile.getUserName()  
            : "<No user profile>";
    }
}
